<?php

/* 
   This class is a port of the Javascript Diff3 algorithm by
      Tony Garnock-Jones <tonyg@lshift.net>
      LShift Ltd. <query@lshift.net>
 */
class Diff {
    public static function longest_common_subsequence($file1, $file2) {
        /* Text diff algorithm following Hunt and McIlroy 1976.
         * J. W. Hunt and M. D. McIlroy, An algorithm for differential file
         * comparison, Bell Telephone Laboratories CSTR #41 (1976)
         * http://www.cs.dartmouth.edu/~doug/
         *
         * Expects two arrays of strings.
         */

        $equivalenceClasses = array();
        for ($j = 0; $j < count($file2); $j++) {
            $line = $file2[$j];
            if (isset($equivalenceClasses[$line])) {
                $equivalenceClasses[$line][] = $j;
            } else {
                $equivalenceClasses[$line] = array($j);
            }
        }

        $candidates = array(
           array(
              "file1index" => -1,
              "file2index" => -1,
              "chain" => null
           )
        );

        for ($i = 0; $i < count($file1); $i++) {
            $line = $file1[$i];
            $file2indices = isset($equivalenceClasses[$line])? $equivalenceClasses[$line] : array();

            $r = 0;
            $c = $candidates[0];

            for ($jX = 0; $jX < count($file2indices); $jX++) {
                $j = $file2indices[$jX];

                for ($s = 0; $s < count($candidates); $s++) {
                    if (($candidates[$s]['file2index'] < $j) &&
                        (($s == count($candidates) - 1) ||
                         ($candidates[$s + 1]['file2index'] > $j)))
                        break;
                }

                if ($s < count($candidates)) {
                    $newCandidate = array("file1index" => $i,
                                    "file2index" => $j,
                                    "chain" => $candidates[$s]);
                    if ($r == count($candidates)) {
                        $candidates[] = $c;
                    } else {
                        $candidates[$r] = $c;
                    }
                    $r = $s + 1;
                    $c = $newCandidate;
                    if ($r == count($candidates)) {
                        break; // no point in examining further (j)s
                    }
                }
            }

            $candidates[$r] = $c;
        }

        // At this point, we know the LCS: it's in the reverse of the
        // linked-list through .chain of
        // candidates[candidates.length - 1].

        return $candidates[count($candidates) - 1];
    }

    public static function processCommon(&$common, &$result) {
          if (count($common['common'])) {
             $common['common'] = array_reverse($common['common']);
             $result[] = $common;
             $common = array('common' => array());
          }
    }

    public static function diff_comm($file1, $file2) {
        // We apply the LCS to build a "comm"-style picture of the
        // differences between file1 and file2.

        $result = array();
        $tail1 = count($file1);
        $tail2 = count($file2);
        $common = array("common" => array());

        
        for ($candidate = Diff::longest_common_subsequence($file1, $file2);
             $candidate !== null;
             $candidate = $candidate['chain'])
        {
            $different = array("file1" => array(), "file2" => array());

            while (--$tail1 > $candidate['file1index']) {
                $different['file1'][] = $file1[$tail1];
            }

            while (--$tail2 > $candidate['file2index']) {
                $different['file2'][] = $file2[$tail2];
            }

            if (count($different['file1']) || count($different['file2'])) {
               Diff::processCommon($common, $result);
                $different['file1'] = array_reverse($different['file1']);
                $different['file2'] = array_reverse($different['file2']);
                $result[] = $different;
            }

            if ($tail1 >= 0) {
                $common['common'][] = $file1[$tail1];
            }
        }

        Diff::processCommon($common, $result);

        $result = array_reverse($result);
        return $result;
    }

    public static function chunkDescription($file, $offset, $length) {
       $chunk = array();
       for ($i = 0; $i < $length; $i++) {
          $chunk[] = $file[$offset + $i];
       }
       return array('offset' => $offset, 'length' => $length, 'chunk' => $chunk);
    }

    public static function diff_patch($file1, $file2) {
        // We apply the LCD to build a JSON representation of a
        // diff(1)-style patch.

        $result = array();
        $tail1 = count($file1);
        $tail2 = count($file2);

        
        for ($candidate = Diff::longest_common_subsequence($file1, $file2);
             $candidate !== null;
             $candidate = $candidate['chain'])
        {
            $mismatchLength1 = $tail1 - $candidate['file1index'] - 1;
            $mismatchLength2 = $tail2 - $candidate['file2index'] - 1;
            $tail1 = $candidate['file1index'];
            $tail2 = $candidate['file2index'];

            if ($mismatchLength1 || $mismatchLength2) {
               $result[] = array(
                  'file1' => Diff::chunkDescription($file1, $candidate['file1index'] + 1, $mismatchLength1),
                  'file2' => Diff::chunkDescription($file2, $candidate['file2index'] + 1, $mismatchLength2),
               );
            }
        }

        $result = array_reverse($result);
        return $result;
    }

    public static function strip_patch($patch) {
       // Takes the output of Diff.diff_patch(), and removes
       // information from it. It can still be used by patch(),
       // below, but can no longer be inverted.
       $newpatch = array();
       for ($i = 0; $i < count($patch); $i++) {
          $chunk = $patch[$i];
          $newpatch[] = array(
             'file1' => array(
                  'offset' => $chunk['file1']['offset'],
                  'length' => count($chunk['file1'])
             ),
             'file2' => array(
                'chunk' => $chunk['file2']['chunk']
             )
          );
       }
       return $newpatch;
    }

    public static function invert_patch($patch) {
        // Takes the output of Diff.diff_patch(), and inverts the
        // sense of it, so that it can be applied to file2 to give
        // file1 rather than the other way around.

        for ($i = 0; i < count($patch); $i++) {
            $chunk = $patch[$i];
            $tmp = $chunk['file1'];
            $chunk['file1'] = $chunk['file2'];
            $chunk['file2'] = $tmp;
        }
    }

    public static function copyCommon($targetOffset, &$commonOffset, &$file, &$result) {
       while ($commonOffset < $targetOffset) {
          $result[] = $file[$commonOffset];
          $commonOffset++;
       }
    }


    public static function patch($file, $patch) {
        // Applies a patch to a file.
        //
        // Given file1 and file2, Diff.patch(file1,
        // Diff.diff_patch(file1, file2)) should give file2.

        $result = array();
        $commonOffset = 0;

        for ($chunkIndex = 0; $chunkIndex < count($patch); $chunkIndex++) {
            $chunk = $patch[$chunkIndex];
            Diff::copyCommon($chunk['file1']['offset'], $commonOffset, $file, $result);
            for ($lineIndex = 0; $lineIndex < count($chunk['file2']['chunk']); $lineIndex++) {
                $result[] = $chunk['file2']['chunk'][lineIndex];
            }
            $commonOffset += count($chunk['file1']);
        }

        Diff::copyCommon(count($file), $commonOffset, $file, $result);
        return $result;
    }

    public static function diff_indices($file1, $file2) {
        // We apply the LCS to give a simple representation of the
        // offsets and lengths of mismatched chunks in the input
        // files. This is used by diff3_merge_indices below.

        $result = array();
        $tail1 = count($file1);
        $tail2 = count($file2);

        for ($candidate = Diff::longest_common_subsequence($file1, $file2);
             $candidate !== null;
             $candidate = $candidate['chain'])
        {
            $mismatchLength1 = $tail1 - $candidate['file1index'] - 1;
            $mismatchLength2 = $tail2 - $candidate['file2index'] - 1;
            $tail1 = $candidate['file1index'];
            $tail2 = $candidate['file2index'];

            if ($mismatchLength1 || $mismatchLength2) {
               $result[] = array(
                  'file1' => array($tail1 + 1, $mismatchLength1),
                  'file2' => array($tail2 + 1, $mismatchLength2),
               );
            }
        }

        $result = array_reverse($result);
        return $result;
    }

    public static function addHunk($h, $side, &$hunks) {
       $hunks[] = array($h['file1'][0], $side, $h['file1'][1], $h['file2'][0], $h['file2'][1]);
    }

    public static function copyCommon2($targetOffset, &$commonOffset, &$result) {
       if ($targetOffset > $commonOffset) {
          $result[] = array(1, $commonOffset, $targetOffset - $commonOffset);
          $commonOffset = $targetOffset;
       }
    }

    public static function diff3_merge_indices($a, $o, $b) {
        // Given three files, A, O, and B, where both A and B are
        // independently derived from O, returns a fairly complicated
        // internal representation of merge decisions it's taken. The
        // interested reader may wish to consult
        //
        // Sanjeev Khanna, Keshav Kunal, and Benjamin C. Pierce. "A
        // Formal Investigation of Diff3." In Arvind and Prasad,
        // editors, Foundations of Software Technology and Theoretical
        // Computer Science (FSTTCS), December 2007.
        //
        // (http://www.cis.upenn.edu/~bcpierce/papers/diff3-short.pdf)
        $m1 = Diff::diff_indices($o, $a);
        $m2 = Diff::diff_indices($o, $b);

        $hunks = array();
        for ($i = 0; $i < count($m1); $i++) { Diff::addHunk($m1[$i], 0, $hunks); }
        for ($i = 0; $i < count($m2); $i++) { Diff::addHunk($m2[$i], 2, $hunks); }
        sort($hunks);

        $result = array();
        $commonOffset = 0;
        
        for ($hunkIndex = 0; $hunkIndex < count($hunks); $hunkIndex++) {
            $firstHunkIndex = $hunkIndex;
            $hunk = $hunks[$hunkIndex];
            $regionLhs = $hunk[0];
            $regionRhs = $regionLhs + $hunk[2];
            while ($hunkIndex < count($hunks) - 1) {
                $maybeOverlapping = $hunks[$hunkIndex + 1];
                $maybeLhs = $maybeOverlapping[0];
                if ($maybeLhs > $regionRhs) break;
                $regionRhs = $maybeLhs + $maybeOverlapping[2];
                $hunkIndex++;
            }

            Diff::copyCommon2($regionLhs, $commonOffset, $result);
            if ($firstHunkIndex == $hunkIndex) {
		// The "overlap" was only one hunk long, meaning that
		// there's no conflict here. Either a and o were the
		// same, or b and o were the same.
                if ($hunk[4] > 0) {
                    $result[] = array($hunk[1], $hunk[3], $hunk[4]);
                }
            } else {
               // A proper conflict. Determine the extents of the
               // regions involved from a, o and b. Effectively merge
               // all the hunks on the left into one giant hunk, and
               // do the same for the right; then, correct for skew
               // in the regions of o that each side changed, and
               // report appropriate spans for the three sides.
               $regions = array(
                  0 => array(count($a), -1, count($o), -1),
                  2 => array(count($b), -1, count($o), -1),
               );
               for ($i = $firstHunkIndex; $i <= $hunkIndex; $i++) {
                  $hunk = $hunks[$i];
                  $side = $hunk[1];
                  $r = $regions[$side];
                  $oLhs = $hunk[0];
                  $oRhs = $oLhs + $hunk[2];
                  $abLhs = $hunk[3];
                  $abRhs = $abLhs + $hunk[4];
                  $r[0] = min($abLhs, $r[0]);
                  $r[1] = max($abRhs, $r[1]);
                  $r[2] = min($oLhs, $r[2]);
                  $r[3] = max($oRhs, $r[3]);
                  $regions[$side] = $r;
               }
               $aLhs = $regions[0][0] + ($regionLhs - $regions[0][2]);
               $aRhs = $regions[0][1] + ($regionRhs - $regions[0][3]);
               $bLhs = $regions[2][0] + ($regionLhs - $regions[2][2]);
               $bRhs = $regions[2][1] + ($regionRhs - $regions[2][3]);
               $result[] = array(
                  -1,
                  $aLhs,      $aRhs      - $aLhs,
                  $regionLhs, $regionRhs - $regionLhs,
                  $bLhs,      $bRhs      - $bLhs
               );
            }
            $commonOffset = $regionRhs;
        }

        Diff::copyCommon2(count($o), $commonOffset, $result);
        return $result;
    }

    public static function flushOk(&$okLines, &$result) {
       if (count($okLines)) {
          $result[] = array('ok' => $okLines);
       }
       $okLines = array();
    }
    public static function pushOk($xs, &$okLines) {
       for ($j = 0; $j < count($xs); $j++) {
          $okLines[] = $xs[$j];
       }
    }

    public static function isTrueConflict($rec, &$a, &$b) {
       if ($rec[2] != $rec[6]) return true;
       $aoff = $rec[1];
       $boff = $rec[5];
       for ($j = 0; $j < $rec[2]; $j++) {
          if ($a[$j + $aoff] != $b[$j + $boff]) return true;
       }
       return false;
    }

    public static function diff3_merge($a, $o, $b, $excludeFalseConflicts) {
        // Applies the output of Diff.diff3_merge_indices to actually
        // construct the merged file; the returned result alternates
        // between "ok" and "conflict" blocks.

        $result = array();
        $files = array($a, $o, $b);
        $indices = Diff::diff3_merge_indices($a, $o, $b);

        $okLines = array();
        
        for ($i = 0; $i < count($indices); $i++) {
           $x = $indices[$i];
           $side = $x[0];
           if ($side == -1) {
              if ($excludeFalseConflicts && !Diff::isTrueConflict($x, $a, $b)) {
                 Diff::pushOk(array_slice($files[0], $x[1], $x[2]), $okLines);
              } else {
                 Diff::flushOk($okLines, $result);
                 $result[] = array(
                    'conflict' => array('a' => array_slice($a, $x[1], $x[2]),
                    'aIndex' => $x[1],
                    'o' => array_slice($o, $x[3], $x[4]),
                    'oIndex' => $x[3],
                    'b' => array_slice($b, $x[5], $x[6]),
                    'bIndex' => $x[5])
                 );
              }
           } else {
              Diff::pushOk(array_slice($files[$side], $x[1], $x[2]), $okLines);
           }
        }

        Diff::flushOk($okLines, $result);
        return $result;
    }
};



function diff3_dig_in($file_last, $orig, $file_current) {
   $new_file_name = preg_replace('/.last$/', ".new", $file_last);
   
   $merger = Diff::diff3_merge(
      explode("\n", str_replace("\r", '', file_get_contents($file_last))),
      explode("\n", str_replace("\r", '', file_get_contents($orig))),
      explode("\n", str_replace("\r", '', file_get_contents($file_current))),
      false
   );

   $nbconflict = 0;
   $lines = array();
   for ($i = 0; $i < count($merger); $i++) {
      $item = $merger[$i];
      if (isset($item['ok'])) {
         $lines = array_merge($lines, $item['ok']);
      } else {
         $c = Diff::diff_comm($item['conflict']['a'], $item['conflict']['b']);
         for ($j = 0; $j < count($c); $j++) {
            $inner = $c[$j];
            if (isset($inner['common'])) {
               $lines = array_merge($lines, $inner['common']);
            } else {
               $nbconflict++;
               $lines = array_merge($lines, array("<<<<<<<<<"), $inner['file1'], array("========="), $inner['file2'], array(">>>>>>>>>"));
            }
         }
      }
   }
   return array(
      'conflict' => $nbconflict,
      'result' => implode("\n", $lines)
   );
}
